using Amx.Core;
using Amx.Logic.Debug;
using Amx.Strings;

namespace Files.Smx
{
    using System;
    using System.IO;
    using System.Collections;

    using Files;

    //using Files.Amx;

    using ICSharpCode.SharpZipLib.Zip.Compression;
    using ICSharpCode.SharpZipLib.Zip.Compression.Streams;

    public class SmxHeader
    {
        //Smx file header
        protected uint sp_hdr_magic;                    /**< Magic number */
        protected UInt16 sp_hdr_version;	            /**< Version code */
        protected SpFileCompression sp_hdr_compression; /**< Compression algorithm */  //(byte)
        protected uint sp_hdr_disksize;	                /**< Size on disk */
        protected uint sp_hdr_imagesize;	            /**< Size in memory */
        protected byte sp_hdr_sections;	                /**< Number of sections */
        protected uint sp_hdr_stringtab;	            /**< Offset to string table */
        protected uint sp_hdr_dataoffs;	                /**< Offset to file proper (any compression starts here) */

        public SmxHeader()
        {

        }

        public SmxHeader(BinaryReader bin)
        {
            this.ReadSpHeader(bin);
        }

        public void ReadSpHeader(BinaryReader bin)
        {
            if (bin != null)
            {
                sp_hdr_magic = bin.ReadUInt32();
                if (sp_hdr_magic == this.MagicNumber)
                {
                    this.sp_hdr_version = bin.ReadUInt16();
                    this.sp_hdr_compression = (SpFileCompression)bin.ReadByte();
                    this.sp_hdr_disksize = bin.ReadUInt32();
                    this.sp_hdr_imagesize = bin.ReadUInt32();
                    this.sp_hdr_sections = bin.ReadByte();
                    this.sp_hdr_stringtab = bin.ReadUInt32();
                    this.sp_hdr_dataoffs = bin.ReadUInt32();
                }
            }
        }

        public bool Valid
        {
            get
            {
                return (sp_hdr_magic == MagicNumber);
            }
        }

        public int MagicNumber
        {
            get
            {
                return 0x53504646; //  S P F F
            }
        }

        public uint Magic
        {
            get
            {
                return sp_hdr_magic;
            }
        }
        public UInt16 Version
        {
            get
            {
                return sp_hdr_version;
            }
        }
        public SpFileCompression Compression
        {
            get
            {
                return sp_hdr_compression;
            }
        }
        public int DiskSize
        {
            get
            {
                return (int)sp_hdr_disksize;
            }
        }
        public int ImageSize
        {
            get
            {
                return (int)sp_hdr_imagesize;
            }
        }
        public byte Sections
        {
            get
            {
                return sp_hdr_sections;
            }
        }
        public uint StringTableOffset
        {
            get
            {
                return sp_hdr_stringtab;
            }
        }
        public uint DataOffset
        {
            get
            {
                return sp_hdr_dataoffs;
            }
        }
    }

    public class SmxFile : IAmxDescription, IAmxInstance, IAmxContainer
    {
        //							    =  S P F F
        // public const int HeaderNumber=0x53504646;
        protected SmxHeader smxHeader;

        protected string path;
        protected SpFileSections[] fileSections; //TODO: better change with other type of store, also add enum to easily acess them
        protected Hashtable stringTable;         //Names forFileSections
        protected Bits bits;

        protected MemoryStream spUncompressedStream;

        private SpFileCodeSection fileCodeSect;
        private int fileCodeOffset;
        private SpFileDataSection fileDataSect;
        private int fileDataOffset;

        private Hashtable filePublicsSect;
        private Hashtable fileNativesSect;
        private Hashtable filePubVarsSect;
        private Hashtable fileTagSect;

        private Hashtable fileNameTable; //Key - UINT

        private SpFileDebugInfoSection fileDebugInfoSect;
        private Hashtable fileDebugFileSect;
        private Hashtable fileDebugLineSect;
        private Hashtable fileDebugSymbolSect;

        private Hashtable fileDebugNameTable; //Key - UINT

        protected Stream amxFile;

        private AmxHeader amxHeader;
        private AmxDebugHeader debugHeader;

        protected PackedStringDecoder decoder;

        public SmxFile(string path)
        {
            FileStream fs = null;
            //BinaryReader bin=null;
            if (File.Exists(path))
            {
                this.path = path;
                try
                {
                    fs = File.Open(path, FileMode.Open, FileAccess.Read);

                    if (fs != null)
                    {
                        BinaryReader reader = new BinaryReader(fs);
                        smxHeader = new SmxHeader(reader);

                        this.ReadSpFileSections(reader);
                        this.ReadSpStringTable(reader);

                        this.UncompressStream(reader);

                        //FileStream fss = null;
                        //fss = File.Open(path + "z", FileMode.OpenOrCreate, FileAccess.Write);
                        //spUncompressedStream.WriteTo(fss);
                        //fss.Close();
                        //fss = null;

                        this.ReadSpSections(spUncompressedStream);

                        bits = fileCodeSect.CellSize;
                        this.FillAmxHeader();
                        this.FillDebugheader();

                        decoder = new PackedStringDecoder(4,PackedType.smxBigEndian);
                    }
                }
                finally
                {
                    if (fs != null)
                    {
                        fs.Close();
                        fs = null;
                    }
                }
            }
        }

        protected virtual void FillAmxHeader()
        {
            var aH = new EmptyAmxHeader();

            aH.WriteAmxVersion = 8;
            aH.WriteFileVersion = 8;
            aH.WriteLength = smxHeader.ImageSize; //BUG: filled with wrong value
            aH.WriteMagic = AmxHeader.AMX_MAGIC;
            aH.WriteBits = bits;

            if (fileCodeSect.GetFlag(SpCodeFlag.SP_FLAG_DEBUG))
            {
                aH.WriteFlags = AMX_FLAGS.DEBUG;
            }
            else
            {
                aH.WriteFlags = (AMX_FLAGS)0;
            }

            aH.WriteCodOffset = fileCodeOffset;
            aH.WriteCodSize = fileCodeSect.CodeSize;
            aH.WriteDatOffset = fileDataOffset;
            aH.WriteDatSize = fileDataSect.DataSize;
            aH.WriteCipOffset = (int)fileCodeSect.Main; //TODO: did without thinking, check that here no errors
            aH.WriteHeaOffset = smxHeader.ImageSize + 4096; //BUG: filled with wrong value, Really calculated from fileDataSect.memsize
            aH.WriteStpOffset = smxHeader.ImageSize; //BUG: filled with wrong value, Really calculated from fileDataSect.memsize

            if (filePubVarsSect.Count > 0)
            {
                AmxNative[] pubvars = new AmxNativeFixed[filePubVarsSect.Count];
                for (int i = 0; i < filePubVarsSect.Count; i++)
                {
                    SpFilePubVarsSection temp = (SpFilePubVarsSection)filePubVarsSect[i];
                    pubvars[i] = new AmxNativeFixed((ValInt)temp.Address, fileNameTable[temp.Name] as string);
                }
                aH.FillPublicVars = pubvars;
            }
            else
            {
                aH.FillPublicVars = new AmxNative[0];
            }

            if (filePublicsSect.Count > 0)
            {
                AmxNative[] pubvars = new AmxNativeFixed[filePublicsSect.Count];
                for (int i = 0; i < filePublicsSect.Count; i++)
                {
                    SpFilePublicsSection temp = (SpFilePublicsSection)filePublicsSect[i];
                    pubvars[i] = new AmxNativeFixed((ValInt)temp.Address, fileNameTable[temp.Name] as string);
                }
                aH.FillPublics = pubvars;
            }
            else
            {
                aH.FillPublics = new AmxNative[0];
            }

            if (fileNativesSect.Count > 0)
            {
                AmxNative[] pubvars = new AmxNativeFixed[fileNativesSect.Count];
                for (int i = 0; i < fileNativesSect.Count; i++)
                {
                    SpFileNativesSection temp = (SpFileNativesSection)fileNativesSect[i];
                    pubvars[i] = new AmxNativeFixed((ValInt)0, fileNameTable[temp.Name] as string);
                }
                aH.FillNatives = pubvars;
            }
            else
            {
                aH.FillNatives = new AmxNative[0];
            }

            if (fileTagSect.Count > 0)
            {
                AmxNative[] pubvars = new AmxNativeFixed[fileTagSect.Count];
                for (int i = 0; i < fileTagSect.Count; i++)
                {
                    SpFileTagSection temp = (SpFileTagSection)fileTagSect[i];
                    pubvars[i] = new AmxNativeFixed((ValInt)temp.TagId, fileNameTable[temp.Name] as string);
                }
                aH.FillTags = pubvars;
            }
            else
            {
                aH.FillTags = new AmxNative[0];
            }

            aH.FillLibraries = new AmxNative[0];

            amxHeader = aH;
        }

        protected virtual void FillDebugheader()
        {
            EmptyAmxDebugHeader dbgHeader = new EmptyAmxDebugHeader();

            dbgHeader.WriteAmxVersion = 8;
            dbgHeader.WriteFileVersion = 8;
            dbgHeader.WriteFlags = 0;   //not used
            //dbgHeader.WriteLength  =
            dbgHeader.WriteNumberOfAutomatons = 0;//not used
            dbgHeader.WriteNumberOfFiles = fileDebugInfoSect.NumberOfFiles;
            dbgHeader.WriteNumberOfLines = fileDebugInfoSect.NumberOfLines;
            dbgHeader.WriteNumberOfStates = 0;//not used
            dbgHeader.WriteNumberOfSymbols = fileDebugInfoSect.NumberOfSymbols;
            dbgHeader.WriteNumberOfTags = fileTagSect.Count;

            SymbolTable files = new SymbolTable();
            foreach (SpFileDebugFileSection file in fileDebugFileSect.Values)
            {
                files.Add((int)file.Address, new SFile((int)file.Address, (string)fileDebugNameTable[file.Name]));
            }
            dbgHeader.FillFileTable = files;

            SymbolTable lines = new SymbolTable();
            foreach (SpFileDebugLineSection line in fileDebugLineSect.Values)
            {
                lines.Add((int)line.Address, new SLine((int)line.Address, (ValInt)line.LineNumber));
            }
            dbgHeader.FillLinesTable = lines;

            SymbolTable globalsymbols = new SymbolTable();
            SymbolTable functionsymbols = new SymbolTable();
            ArrayList localSymbols = new ArrayList();
            foreach (SpFileDebugSymbolSection symbol in fileDebugSymbolSect.Values)
            {
                Symbol symb = null;

                symb = new Symbol((int)symbol.Address, (ValInt)symbol.CodeStart, symbol.VarType, symbol.ScopeClass, fileDebugNameTable[symbol.Name] as string);
                symb.AddExtension(new SymbolTag((int)symbol.Address, (int)symbol.TagId));

                foreach (SpFileDebugArrayDimSection dim in symbol.Dimensions.Values)
                {
                    symb.AddExtension(new SymbolRange((int)symbol.Address, (int)dim.TagId, (int)dim.Dimensions));
                }

                if (symbol.ScopeClass == SymbolFlagClass.local)
                {
                    localSymbols.Add(symb);
                }
                else
                {
                    if (symbol.VarType == SymbolFlagType.function || symbol.VarType == SymbolFlagType.function_ptr)
                    {
                        if (!functionsymbols.Contains((int)symbol.Address))
                            functionsymbols.Add((int)symbol.Address, symb);
                    }
                    else
                    {
                        if (!globalsymbols.Contains((int)symbol.Address))
                            globalsymbols.Add((int)symbol.Address, symb);
                    }
                }
            }
            dbgHeader.FillFunctionSymbolTable = functionsymbols;
            dbgHeader.FillGlobalSymbolTable = globalsymbols;
            dbgHeader.FillLocalSymbolTable = localSymbols;

            ArrayList tags = new ArrayList();
            foreach (SpFileTagSection tag in fileTagSect.Values)
            {
                tags.Add(new SymbolTagRealization(tag.TagId, fileNameTable[tag.Name] as string)); //TODO: check for proper position in Array
            }
            dbgHeader.FillTagTable = tags;

            this.debugHeader = dbgHeader;
        }

        protected virtual void ReadSpStringTable(BinaryReader bin)
        {
            stringTable = new Hashtable(smxHeader.Sections);
            if (bin != null)
            {
                bin.BaseStream.Seek(smxHeader.StringTableOffset, SeekOrigin.Begin); //really no need;

                for (int i = 0; i < smxHeader.Sections; i++)
                {
                    string buf = "";
                    int relativePos = (int)(bin.BaseStream.Position - smxHeader.StringTableOffset);

                    while (bin.BaseStream.Position < bin.BaseStream.Length)
                    {
                        char ch = bin.ReadChar();
                        if (ch == 0)
                            break;
                        buf += (ch);
                    }

                    stringTable.Add(relativePos, buf);
                }
            }
        }

        protected virtual void UncompressStream(BinaryReader reader)
        {
            this.spUncompressedStream = new MemoryStream(smxHeader.ImageSize);

            if (smxHeader.Compression == SpFileCompression.SPFILE_COMPRESSION_NONE)
            {
                reader.BaseStream.Seek(0L, SeekOrigin.Begin);
                spUncompressedStream.Write(reader.ReadBytes(smxHeader.ImageSize), 0, smxHeader.ImageSize); //TODO: never checked - Check It
                spUncompressedStream.Seek(0L, SeekOrigin.Begin);
                reader.BaseStream.Seek(0L, SeekOrigin.Begin);
            }
            else if (smxHeader.Compression == SpFileCompression.SPFILE_COMPRESSION_GZ)
            {
                reader.BaseStream.Seek(0L, SeekOrigin.Begin);
                spUncompressedStream.Write(reader.ReadBytes((int)fileSections[0].DataOffset), 0, (int)fileSections[0].DataOffset);

                //spUncompressedStream.Seek(fileSections[0].DataOffset, SeekOrigin.Begin);    //Skip header, and not write it into stream //TODO: Refactor whole code //BUG: with not filled header in stream

                //reader.BaseStream.Seek(fileSections[0].DataOffset, SeekOrigin.Begin);
                MemoryStream buffer = new MemoryStream((int)(smxHeader.DiskSize - fileSections[0].DataOffset)); //really - fileSections[0].DataOffset

                buffer.Write(reader.ReadBytes((int)(smxHeader.DiskSize - fileSections[0].DataOffset)), 0, (int)(smxHeader.DiskSize - fileSections[0].DataOffset)); //TODO: never checked - Check It

                buffer.Seek(0L, SeekOrigin.Begin); // get back the beginning of the buffer before we try to read.
                {
                    InflaterInputStream inflater = new InflaterInputStream(buffer);
                    byte[] buf = new byte[1024];
                    int read = 0;
                    while (inflater.Available != 0)
                    {
                        read = inflater.Read(buf, 0, buf.Length);
                        //fss.Write(buf,0,read);
                        spUncompressedStream.Write(buf, 0, read);
                    }
                    inflater = null;
                }
                buffer.Close();
                buffer = null;
                spUncompressedStream.Seek(0L, SeekOrigin.Begin);
                reader.BaseStream.Seek(0L, SeekOrigin.Begin);
            }
            else
            {
                throw new ApplicationException("Error 5 : Unknown compression Type");
            }
        }

        protected virtual void ReadSpSections(MemoryStream bin)
        {
            foreach (SpFileSections spfs in fileSections)
            {
                spUncompressedStream.Seek(spfs.DataOffset, SeekOrigin.Begin);

                string sectionName = (string)stringTable[(int)spfs.NameOffset];
                #region code
                if (sectionName == ".code")
                {
                    fileCodeSect = new SpFileCodeSection();
                    fileCodeSect.Read(new BinaryReader(bin));
                    fileCodeOffset = (int)(spfs.DataOffset + fileCodeSect.Code); //or bin.pos
                }
                #endregion
                #region Data
                if (sectionName == ".data")
                {
                    fileDataSect = new SpFileDataSection();
                    fileDataSect.Read(new BinaryReader(bin));
                    fileDataOffset = (int)(spfs.DataOffset + fileDataSect.Data);//or bin.pos
                }
                #endregion
                #region Publics
                if (sectionName == ".publics")
                { // ,    
                    int i = 0;
                    filePublicsSect = new Hashtable();
                    while (spfs.DataOffset + spfs.Size > spUncompressedStream.Position)
                    {
                        var temp = new SpFilePublicsSection();
                        temp.Read(new BinaryReader(bin));
                        filePublicsSect.Add(i, temp);
                        i++;
                    }
                }
                #endregion
                #region PubVars
                if (sectionName == ".pubvars")
                {
                    int i = 0;
                    filePubVarsSect = new Hashtable();
                    while (spfs.DataOffset + spfs.Size > spUncompressedStream.Position)
                    {
                        var temp = new SpFilePubVarsSection();
                        temp.Read(new BinaryReader(bin));
                        filePubVarsSect.Add(i, temp);
                        i++;
                    }
                }
                #endregion
                #region Natvies
                if (sectionName == ".natives")
                { //   ,    -                       
                    int i = 0;
                    fileNativesSect = new Hashtable();
                    while (spfs.DataOffset + spfs.Size > spUncompressedStream.Position)
                    {
                        var temp = new SpFileNativesSection();
                        temp.Read(new BinaryReader(bin));
                        fileNativesSect.Add(i, temp);
                        i++;
                    }
                }
                #endregion
                #region Tags
                if (sectionName == ".tags")
                {// ,    
                    int i = 0;
                    fileTagSect = new Hashtable();
                    while (spfs.DataOffset + spfs.Size > spUncompressedStream.Position)
                    {
                        var temp = new SpFileTagSection();
                        temp.Read(new BinaryReader(bin));
                        fileTagSect.Add(i, temp);
                        i++;
                    } // ????? # of tags
                }
                #endregion
                if (sectionName == ".names")
                {
                    fileNameTable = new Hashtable();

                    //int i = 0;
                    while (spfs.DataOffset + spfs.Size > spUncompressedStream.Position)
                    {
                        BinaryReader br = new BinaryReader(bin, System.Text.Encoding.ASCII);
                        string buf = "";
                        long position = spUncompressedStream.Position - spfs.DataOffset;
                        while (spUncompressedStream.Position < spUncompressedStream.Length)
                        {
                            char ch = br.ReadChar();
                            if (ch == 0)
                                break;
                            buf += (ch);
                        }

                        fileNameTable.Add((uint)position, buf);
                        //Possible wrong some last Entries...
                    }
                }
                #region Debug Files
                if (sectionName == ".dbg.files")
                { // ,    
                    int i = 0;
                    fileDebugFileSect = new Hashtable();
                    while (spfs.DataOffset + spfs.Size > spUncompressedStream.Position)
                    {
                        var temp = new SpFileDebugFileSection();
                        temp.Read(new BinaryReader(bin));
                        fileDebugFileSect.Add(i, temp);
                        i++;
                    }
                }
                #endregion
                #region Debug Lines
                if (sectionName == ".dbg.lines")
                {
                    int i = 0;
                    fileDebugLineSect = new Hashtable();
                    while (spfs.DataOffset + spfs.Size > spUncompressedStream.Position)
                    {
                        var temp = new SpFileDebugLineSection();
                        temp.Read(new BinaryReader(bin));
                        fileDebugLineSect.Add(i, temp);
                        i++;
                    }
                }
                #endregion
                #region Debug Symbols
                if (sectionName == ".dbg.symbols")
                {// ,     -        ...
                    int i = 0;
                    fileDebugSymbolSect = new Hashtable();
                    while (spfs.DataOffset + spfs.Size > spUncompressedStream.Position)
                    {
                        var temp = new SpFileDebugSymbolSection();
                        temp.Read(new BinaryReader(bin));
                        fileDebugSymbolSect.Add(i, temp);
                        i++;
                    }
                }
                #endregion
                #region Debug Info
                if (sectionName == ".dbg.info")
                { // 
                    fileDebugInfoSect = new SpFileDebugInfoSection();
                    fileDebugInfoSect.Read(new BinaryReader(bin));
                }
                #endregion
                if (sectionName == ".dbg.strings")
                {
                    fileDebugNameTable = new Hashtable();

                    //int i = 0;
                    while (spfs.DataOffset + spfs.Size > spUncompressedStream.Position)
                    {
                        BinaryReader br = new BinaryReader(bin, System.Text.Encoding.ASCII);
                        string buf = "";
                        long position = spUncompressedStream.Position - spfs.DataOffset;
                        while (spUncompressedStream.Position < spUncompressedStream.Length)
                        {
                            char ch = br.ReadChar();
                            if (ch == 0)
                                break;
                            buf += (ch);
                        }

                        fileDebugNameTable.Add((uint)position, buf);
                        //i++;
                    }
                }
                if (sectionName == ".dbg.natives")
                { //    
                    //fileDebugNtrnfoSect = new SpFileDebugInfoSection();
                    //fileDebugInfoSect.Read(new AmxStream(bin));
                }
            }
        }

        protected virtual void ReadSpFileSections(BinaryReader bin)
        {
            fileSections = new SpFileSections[smxHeader.Sections];
            if (bin != null)
            {
                for (int i = 0; i < smxHeader.Sections; i++)
                {
                    var newSection = new SpFileSections();
                    newSection.Read(bin);
                    fileSections[i] = newSection;
                }
            }
        }

        public virtual int MagicNumber
        {
            get
            {
                return 0x53504646; //  S P F F
            }
        }

        public virtual IAmxInstance GetFile(IAmxDescription description)
        {
            if (description.Valid && description.Bits == bits)
            {
                return this;
            }
            else
            {
                return null;
            }
        }


        #region IAmxContainer Members

        public virtual int Count
        {
            get
            {
                return 1;
            }

        }

        public virtual string Path
        {
            get
            {
                return path;
            }
        }

        public virtual string Name
        {
            get
            {
                return System.IO.Path.GetFileNameWithoutExtension(path);
            }
        }

        public virtual AmxStream GetStream(IAmxDescription description)
        {
            if (description.Bits == bits && description.Valid)
            {
                return this.GetStream();
            }
            return null;
        }

        #endregion

        #region IEnumerable Members

        public IEnumerator GetEnumerator()
        {
            throw new ApplicationException("1000: Implement me");

            return new SmxFileEnumerator(); //TODO: implement me
        }

        #endregion

        #region IAmxInstance Members

        public AmxHeader Header
        {
            get { return amxHeader; }
        }

        public AmxDebugHeader DebugHeader
        {
            get { return debugHeader; }
        }

        public string InfoString
        {
            get
            {
                string retval = null;

                retval = this.Name + ", " + this.Bits.ToString().Replace("_", "") + " bit";

                return retval;
            }
        }

        public AmxStream GetStream()
        {
            MemoryStream temp = new MemoryStream((int)spUncompressedStream.Length);
            spUncompressedStream.WriteTo(temp);
            spUncompressedStream.Seek(0L, SeekOrigin.Begin);
            temp.Seek(0L, SeekOrigin.Begin);
            return new AmxStream(temp);
        }

        public IAmxContainer Container
        {
            get { return this; }
        }

        public IAmxDescription Description
        {
            get { return this; }
        }

        #endregion

        #region IAmxDescription Members

        public Bits Bits
        {
            get { return bits; } 
        }

        public bool Valid
        {            
            get
            {
                if (smxHeader != null)
                {
                    return smxHeader.Valid;
                }
                else
                {
                    return false;
                }
            }        
        }

        public int Start
        {
            get { return 0; }
        }

        public int Length
        {
            get { return smxHeader.DiskSize; } 
        }

        #endregion
    }

    public class SmxFileEnumerator : IEnumerator
    {//TODO: Implement me!
        private SmxFile file;
        bool seen;

        public SmxFileEnumerator()
        {
        }

        public SmxFileEnumerator(SmxFile file)
        {
            this.file = file;
            this.Reset();
        }

        #region IEnumerator Members

        public void Reset()
        {
            seen = false;
        }

        public object Current
        {
            get
            {
                if (!seen)
                {
                    return file;
                }
                else
                {
                    throw new InvalidOperationException();
                }
            }
        }

        public bool MoveNext()
        {
            if (!seen)
            {
                seen = true;
                return seen;
            }
            else
            {
                return false;
            }
        }

        #endregion

    }
    
    public class SmxFiletype : IFiletype
    {
        #region IFiletype Members

        public string FriendyName
        {
            get
            {
                return "Smx";
            }
        }

        public string Description
        {
            get
            {
                return "Source Mod file format";
            }
        }

        public string Name
        {
            get
            {
                return "Smx";
            }
        }

        public string Extension
        {
            get
            {

                return ".smx";
            }
        }

        public string FilterText
        {
            get
            {
                return "Source Mod Files (*.smx)";
            }
        }

        public string FilterMask
        {
            get
            {
                return "*.smx";
            }
        }


        public IAmxContainer GetContainer(string path)
        {
            return new SmxFile(path);
        }

        public IAmxInstance GetInstance(string path, Bits bits)
        {
            IAmxInstance inst = null;
            IAmxContainer cont = this.GetContainer(path);
            foreach (IAmxDescription desc in cont)
            {
                if (desc.Bits == bits && desc.Valid)
                {
                    inst = cont.GetFile(desc);
                    break;
                }
            }
            return inst;            
        }

        #endregion
    }
}